<?php

/**
* @package   s9e\TextFormatter
* @copyright Copyright (c) 2010-2021 The s9e authors
* @license   http://www.opensource.org/licenses/mit-license.php The MIT License
*/
namespace s9e\TextFormatter\Configurator\JavaScript;

use RuntimeException;
use s9e\TextFormatter\Configurator\Helpers\RegexpParser;

/**
* @todo create a method that replaces capturing subpatterns with non-capturing subpatterns, and perhaps even no subpattern at all. Unless backreferences are used
*/
abstract class RegexpConvertor
{
	/**
	* Regexp that matches Unicode properties escape sequences
	*/
	protected static $unicodePropsRegexp = '((?<!\\\\)((?:\\\\\\\\)*+)\\\\([Pp](?:L[mo]?|N[dlo]?|P[c-fios]?|S[ckmo]?|Z[lps]?|\\{(?:L[mo]?|N[dlo]?|P[c-fios]?|S[ckmo]?|Z[lps]?|\\^(?:L[mo]?|N[dlo]?|P[c-fios]?|S[ckmo]?|Z[lps]?))})))';

	/**
	* Convert a PCRE regexp to a JavaScript regexp
	*
	* @param  string $regexp   PCRE regexp
	* @param  bool   $isGlobal Whether the global flag should be set
	* @return Code             JavaScript regexp
	*/
	public static function toJS($regexp, $isGlobal = false)
	{
		$regexpInfo = RegexpParser::parse($regexp);
		$dotAll     = (strpos($regexpInfo['modifiers'], 's') !== false);

		$regexp = '';
		$pos = 0;

		foreach ($regexpInfo['tokens'] as $tok)
		{
			$regexp .= self::convertUnicodeCharacters(
				substr($regexpInfo['regexp'], $pos, $tok['pos'] - $pos),
				false,
				$dotAll
			);

			$callback = get_called_class() . '::serialize' . ucfirst($tok['type']);
			if (!is_callable($callback))
			{
				throw new RuntimeException("Unsupported token type '" . $tok['type'] . "'");
			}
			$regexp .= $callback($tok);

			$pos = $tok['pos'] + $tok['len'];
		}
		$regexp .= self::convertUnicodeCharacters(substr($regexpInfo['regexp'], $pos), false, $dotAll);

		if ($regexpInfo['delimiter'] !== '/')
		{
			$regexp = preg_replace('#(?<!\\\\)((?:\\\\\\\\)*+)/#', '$1\\/', $regexp);
		}

		$modifiers = preg_replace('#[^im]#', '', $regexpInfo['modifiers']);
		if ($isGlobal)
		{
			$modifiers .= 'g';
		}

		if ($regexp === '')
		{
			$regexp = '(?:)';
		}

		return '/' . self::escapeLineTerminators($regexp) . '/' . $modifiers;
	}

	/**
	* Replace Unicode characters and properties in a string
	*
	* NOTE: does not support \X
	*
	* @link http://docs.php.net/manual/en/regexp.reference.unicode.php
	*
	* @param  string $str              Original string
	* @param  bool   $inCharacterClass Whether this string is in a character class
	* @param  bool   $dotAll           Whether PCRE_DOTALL is set
	* @return string                   Modified string
	*/
	protected static function convertUnicodeCharacters($str, $inCharacterClass, $dotAll)
	{
		// Replace \x{....} with \u.... -- Note that only BMP characters are supported
		$str = preg_replace('((?<!\\\\)(?:\\\\\\\\)*\\K\\\\x\\{([0-9a-f]{4})\\})i', '\\u$1', $str);

		// Unfold Unicode properties such as \pL
		$str = self::unfoldUnicodeProperties($str, $inCharacterClass, $dotAll);

		return $str;
	}

	/**
	* Escape line terminators in given regexp
	*
	* @param  string $regexp Original regexp
	* @return string         Modified regexp
	*/
	protected static function escapeLineTerminators($regexp)
	{
		return preg_replace_callback(
			"/(\\\\*)([\\r\\n]|\xE2\x80\xA8|\xE2\x80\xA9)/",
			function ($m)
			{
				$table = [
					"\r" => '\\r',
					"\n" => '\\n',
					"\xE2\x80\xA8" => '\\u2028',
					"\xE2\x80\xA9" => '\\u2029'
				];

				// Ensure we have an even number of backslashes
				if (strlen($m[1]) & 1)
				{
					$m[1] .= '\\';
				}

				return $m[1] . $table[$m[2]];
			},
			$regexp
		);
	}

	protected static function serializeOption(array $tok): string
	{
		if ($tok['options'] !== 'J')
		{
			throw new RuntimeException('Regexp options are not supported');
		}

		return '';
	}

	protected static function serializeCapturingSubpatternStart(array $tok): string
	{
		return '(';
	}

	protected static function serializeNonCapturingSubpatternStart(array $tok): string
	{
		if (!empty($tok['options']))
		{
			throw new RuntimeException('Subpattern options are not supported');
		}

		return '(?:';
	}

	protected static function serializeCapturingSubpatternEnd(array $tok): string
	{
		return ')' . substr($tok['quantifiers'], 0, 1);
	}

	protected static function serializeNonCapturingSubpatternEnd(array $tok): string
	{
		return static::serializeCapturingSubpatternEnd($tok);
	}

	protected static function serializeCharacterClass(array $tok): string
	{
		return '[' . self::convertUnicodeCharacters($tok['content'], true, false) . ']' . substr($tok['quantifiers'], 0, 1);
	}

	protected static function serializeLookaheadAssertionStart(array $tok): string
	{
		return '(?=';
	}

	protected static function serializeNegativeLookaheadAssertionStart(array $tok): string
	{
		return '(?!';
	}

	protected static function serializeLookaheadAssertionEnd(array $tok): string
	{
		return ')';
	}

	protected static function serializeNegativeLookaheadAssertionEnd(array $tok): string
	{
		return ')';
	}

	/**
	* Replace Unicode properties in a string
	*
	* NOTE: does not support \X
	*
	* @link http://docs.php.net/manual/en/regexp.reference.unicode.php
	*
	* @param  string $str              Original string
	* @param  bool   $inCharacterClass Whether this string is in a character class
	* @param  bool   $dotAll           Whether PCRE_DOTALL is set
	* @return string                   Modified string
	*/
	protected static function unfoldUnicodeProperties($str, $inCharacterClass, $dotAll)
	{
		$unicodeProps = self::$unicodeProps;

		$str = preg_replace_callback(
			self::$unicodePropsRegexp,
			function ($m) use ($inCharacterClass, $unicodeProps)
			{
				$propName = preg_replace('#[\\{\\}]#', '', $m[2]);

				if ($propName[1] === '^')
				{
					// Replace p^L with PL
					$propName = (($propName[0] === 'p') ? 'P' : 'p') . substr($propName, 2);
				}

				return (($inCharacterClass) ? '' : '[')
				     . $unicodeProps[$propName]
				     . (($inCharacterClass) ? '' : ']');
			},
			$str
		);

		if ($dotAll)
		{
			$str = preg_replace(
				'#(?<!\\\\)((?:\\\\\\\\)*+)\\.#',
				'$1[\\s\\S]',
				$str
			);
		}

		return $str;
	}

	/**
	* Ranges to be used in JavaScript regexps in place of PCRE's Unicode properties
	*/
	protected static $unicodeProps = [
		'PL' => '\\u0000-@\\[-`\\{-\\u00A9\\u00AB-\\u00B4\\u00B6-\\u00B9\\u00BB-\\u00BF\\u02C2-\\u02C5\\u02D2-\\u02DF\\u02E5-\\u02EB\\u02EF-\\u036F\\u0378\\u0379\\u0380-\\u0385\\u0482-\\u0489\\u0557\\u0558\\u055A-\\u055F\\u0589-\\u05CF\\u05EB-\\u05EE\\u05F3-\\u061F\\u064B-\\u066D\\u06D6-\\u06E4\\u06E7-\\u06ED\\u06F0-\\u06F9\\u06FD\\u06FE\\u0700-\\u070F\\u0730-\\u074C\\u07A6-\\u07B0\\u07B2-\\u07C9\\u07EB-\\u07F3\\u07F6-\\u07F9\\u07FB-\\u07FF\\u0816-\\u0819\\u081B-\\u0823\\u0825-\\u0827\\u0829-\\u083F\\u0859-\\u085F\\u086B-\\u089F\\u08C8-\\u0903\\u093A-\\u093C\\u093E-\\u094F\\u0951-\\u0957\\u0962-\\u0970\\u0981-\\u0984\\u098D\\u098E\\u0991\\u0992\\u09B3-\\u09B5\\u09BA-\\u09BC\\u09BE-\\u09CD\\u09CF-\\u09DB\\u09E2-\\u09EF\\u09F2-\\u09FB\\u09FD-\\u0A04\\u0A0B-\\u0A0E\\u0A11\\u0A12\\u0A3A-\\u0A58\\u0A5F-\\u0A71\\u0A75-\\u0A84\\u0ABA-\\u0ABC\\u0ABE-\\u0ACF\\u0AD1-\\u0ADF\\u0AE2-\\u0AF8\\u0AFA-\\u0B04\\u0B0D\\u0B0E\\u0B11\\u0B12\\u0B3A-\\u0B3C\\u0B3E-\\u0B5B\\u0B62-\\u0B70\\u0B72-\\u0B82\\u0B8B-\\u0B8D\\u0B96-\\u0B98\\u0BA0-\\u0BA2\\u0BA5-\\u0BA7\\u0BAB-\\u0BAD\\u0BBA-\\u0BCF\\u0BD1-\\u0C04\\u0C3A-\\u0C3C\\u0C3E-\\u0C57\\u0C5B-\\u0C5F\\u0C62-\\u0C7F\\u0C81-\\u0C84\\u0CBA-\\u0CBC\\u0CBE-\\u0CDD\\u0CE2-\\u0CF0\\u0CF3-\\u0D03\\u0D3B\\u0D3C\\u0D3E-\\u0D4D\\u0D4F-\\u0D53\\u0D57-\\u0D5E\\u0D62-\\u0D79\\u0D80-\\u0D84\\u0D97-\\u0D99\\u0DBE\\u0DBF\\u0DC7-\\u0E00\\u0E34-\\u0E3F\\u0E47-\\u0E80\\u0EB4-\\u0EBC\\u0EBE\\u0EBF\\u0EC7-\\u0EDB\\u0EE0-\\u0EFF\\u0F01-\\u0F3F\\u0F6D-\\u0F87\\u0F8D-\\u0FFF\\u102B-\\u103E\\u1040-\\u104F\\u1056-\\u1059\\u105E-\\u1060\\u1062-\\u1064\\u1067-\\u106D\\u1071-\\u1074\\u1082-\\u108D\\u108F-\\u109F\\u10C8-\\u10CC\\u10CE\\u10CF\\u124E\\u124F\\u125E\\u125F\\u128E\\u128F\\u12B6\\u12B7\\u12C6\\u12C7\\u1316\\u1317\\u135B-\\u137F\\u1390-\\u139F\\u13F6\\u13F7\\u13FE-\\u1400\\u166D\\u166E\\u169B-\\u169F\\u16EB-\\u16F0\\u16F9-\\u16FF\\u1712-\\u171F\\u1732-\\u173F\\u1752-\\u175F\\u1771-\\u177F\\u17B4-\\u17D6\\u17D8-\\u17DB\\u17DD-\\u181F\\u1879-\\u187F\\u1885\\u1886\\u18AB-\\u18AF\\u18F6-\\u18FF\\u191F-\\u194F\\u196E\\u196F\\u1975-\\u197F\\u19AC-\\u19AF\\u19CA-\\u19FF\\u1A17-\\u1A1F\\u1A55-\\u1AA6\\u1AA8-\\u1B04\\u1B34-\\u1B44\\u1B4C-\\u1B82\\u1BA1-\\u1BAD\\u1BB0-\\u1BB9\\u1BE6-\\u1BFF\\u1C24-\\u1C4C\\u1C50-\\u1C59\\u1C7E\\u1C7F\\u1C89-\\u1C8F\\u1CBB\\u1CBC\\u1CC0-\\u1CE8\\u1CF7-\\u1CF9\\u1CFB-\\u1CFF\\u1DC0-\\u1DFF\\u1F16\\u1F17\\u1F1E\\u1F1F\\u1F46\\u1F47\\u1F4E\\u1F4F\\u1F7E\\u1F7F\\u1FBF-\\u1FC1\\u1FCD-\\u1FCF\\u1FD4\\u1FD5\\u1FDC-\\u1FDF\\u1FED-\\u1FF1\\u1FFD-\\u2070\\u2072-\\u207E\\u2080-\\u208F\\u209D-\\u2101\\u2103-\\u2106\\u2108\\u2109\\u2116-\\u2118\\u211E-\\u2123\\u213A\\u213B\\u2140-\\u2144\\u214A-\\u214D\\u214F-\\u2182\\u2185-\\u2BFF\\u2CE5-\\u2CEA\\u2CEF-\\u2CF1\\u2CF4-\\u2CFF\\u2D28-\\u2D2C\\u2D2E\\u2D2F\\u2D68-\\u2D6E\\u2D70-\\u2D7F\\u2D97-\\u2D9F\\u2DDF-\\u2E2E\\u2E30-\\u3004\\u3007-\\u3030\\u3036-\\u303A\\u303D-\\u3040\\u3097-\\u309C\\u3100-\\u3104\\u318F-\\u319F\\u31C0-\\u31EF\\u3200-\\u33FF\\u4DC0-\\u4DFF\\u9FFD-\\u9FFF\\uA48D-\\uA4CF\\uA4FE\\uA4FF\\uA60D-\\uA60F\\uA620-\\uA629\\uA62C-\\uA63F\\uA66F-\\uA67E\\uA69E\\uA69F\\uA6E6-\\uA716\\uA720\\uA721\\uA789\\uA78A\\uA7C0\\uA7C1\\uA7CB-\\uA7F4\\uA823-\\uA83F\\uA874-\\uA881\\uA8B4-\\uA8F1\\uA8F8-\\uA8FA\\uA8FF-\\uA909\\uA926-\\uA92F\\uA947-\\uA95F\\uA97D-\\uA983\\uA9B3-\\uA9CE\\uA9D0-\\uA9DF\\uA9F0-\\uA9F9\\uAA29-\\uAA3F\\uAA4C-\\uAA5F\\uAA77-\\uAA79\\uAA7B-\\uAA7D\\uAAB2-\\uAAB4\\uAAB7\\uAAB8\\uAABE\\uAABF\\uAAC3-\\uAADA\\uAADE\\uAADF\\uAAEB-\\uAAF1\\uAAF5-\\uAB00\\uAB07\\uAB08\\uAB0F\\uAB10\\uAB17-\\uAB1F\\uAB6A-\\uAB6F\\uABE3-\\uABFF\\uD7A4-\\uD7AF\\uD7C7-\\uD7CA\\uD7FC-\\uF8FF\\uFA6E\\uFA6F\\uFADA-\\uFAFF\\uFB07-\\uFB12\\uFB18-\\uFB1C\\uFBB2-\\uFBD2\\uFD3E-\\uFD4F\\uFD90\\uFD91\\uFDC8-\\uFDEF\\uFDFC-\\uFE6F\\uFEFD-\\uFF20\\uFF3B-\\uFF40\\uFF5B-\\uFF65\\uFFBF-\\uFFC1\\uFFC8\\uFFC9\\uFFD0\\uFFD1\\uFFD8\\uFFD9\\uFFDD-\\uFFFF',
		'PLm' => '\\u0000-\\u02AF\\u02C2-\\u02C5\\u02D2-\\u02DF\\u02E5-\\u02EB\\u02EF-\\u0373\\u0375-\\u0379\\u037B-\\u0558\\u055A-\\u063F\\u0641-\\u06E4\\u06E7-\\u07F3\\u07F6-\\u07F9\\u07FB-\\u0819\\u081B-\\u0823\\u0825-\\u0827\\u0829-\\u0970\\u0972-\\u0E45\\u0E47-\\u0EC5\\u0EC7-\\u10FB\\u10FD-\\u17D6\\u17D8-\\u1842\\u1844-\\u1AA6\\u1AA8-\\u1C77\\u1C7E-\\u1D2B\\u1D6B-\\u1D77\\u1D79-\\u1D9A\\u1DC0-\\u2070\\u2072-\\u207E\\u2080-\\u208F\\u209D-\\u2C7B\\u2C7E-\\u2D6E\\u2D70-\\u2E2E\\u2E30-\\u3004\\u3006-\\u3030\\u3036-\\u303A\\u303C-\\u309C\\u309F-\\u30FB\\u30FF-\\uA014\\uA016-\\uA4F7\\uA4FE-\\uA60B\\uA60D-\\uA67E\\uA680-\\uA69B\\uA69E-\\uA716\\uA720-\\uA76F\\uA771-\\uA787\\uA789-\\uA7F7\\uA7FA-\\uA9CE\\uA9D0-\\uA9E5\\uA9E7-\\uAA6F\\uAA71-\\uAADC\\uAADE-\\uAAF2\\uAAF5-\\uAB5B\\uAB60-\\uAB68\\uAB6A-\\uFF6F\\uFF71-\\uFF9D\\uFFA0-\\uFFFF',
		'PLo' => '\\u0000-\\u00A9\\u00AB-\\u00B9\\u00BB-\\u01BA\\u01BC-\\u01BF\\u01C4-\\u0293\\u0295-\\u05CF\\u05EB-\\u05EE\\u05F3-\\u061F\\u064B-\\u066D\\u06D6-\\u06ED\\u06F0-\\u06F9\\u06FD\\u06FE\\u0700-\\u070F\\u0730-\\u074C\\u07A6-\\u07B0\\u07B2-\\u07C9\\u07EB-\\u07FF\\u0816-\\u083F\\u0859-\\u085F\\u086B-\\u089F\\u08C8-\\u0903\\u093A-\\u093C\\u093E-\\u094F\\u0951-\\u0957\\u0962-\\u0971\\u0981-\\u0984\\u098D\\u098E\\u0991\\u0992\\u09B3-\\u09B5\\u09BA-\\u09BC\\u09BE-\\u09CD\\u09CF-\\u09DB\\u09E2-\\u09EF\\u09F2-\\u09FB\\u09FD-\\u0A04\\u0A0B-\\u0A0E\\u0A11\\u0A12\\u0A3A-\\u0A58\\u0A5F-\\u0A71\\u0A75-\\u0A84\\u0ABA-\\u0ABC\\u0ABE-\\u0ACF\\u0AD1-\\u0ADF\\u0AE2-\\u0AF8\\u0AFA-\\u0B04\\u0B0D\\u0B0E\\u0B11\\u0B12\\u0B3A-\\u0B3C\\u0B3E-\\u0B5B\\u0B62-\\u0B70\\u0B72-\\u0B82\\u0B8B-\\u0B8D\\u0B96-\\u0B98\\u0BA0-\\u0BA2\\u0BA5-\\u0BA7\\u0BAB-\\u0BAD\\u0BBA-\\u0BCF\\u0BD1-\\u0C04\\u0C3A-\\u0C3C\\u0C3E-\\u0C57\\u0C5B-\\u0C5F\\u0C62-\\u0C7F\\u0C81-\\u0C84\\u0CBA-\\u0CBC\\u0CBE-\\u0CDD\\u0CE2-\\u0CF0\\u0CF3-\\u0D03\\u0D3B\\u0D3C\\u0D3E-\\u0D4D\\u0D4F-\\u0D53\\u0D57-\\u0D5E\\u0D62-\\u0D79\\u0D80-\\u0D84\\u0D97-\\u0D99\\u0DBE\\u0DBF\\u0DC7-\\u0E00\\u0E34-\\u0E3F\\u0E46-\\u0E80\\u0EB4-\\u0EBC\\u0EBE\\u0EBF\\u0EC5-\\u0EDB\\u0EE0-\\u0EFF\\u0F01-\\u0F3F\\u0F6D-\\u0F87\\u0F8D-\\u0FFF\\u102B-\\u103E\\u1040-\\u104F\\u1056-\\u1059\\u105E-\\u1060\\u1062-\\u1064\\u1067-\\u106D\\u1071-\\u1074\\u1082-\\u108D\\u108F-\\u10FF\\u124E\\u124F\\u125E\\u125F\\u128E\\u128F\\u12B6\\u12B7\\u12C6\\u12C7\\u1316\\u1317\\u135B-\\u137F\\u1390-\\u1400\\u166D\\u166E\\u169B-\\u169F\\u16EB-\\u16F0\\u16F9-\\u16FF\\u1712-\\u171F\\u1732-\\u173F\\u1752-\\u175F\\u1771-\\u177F\\u17B4-\\u17DB\\u17DD-\\u181F\\u1879-\\u187F\\u1885\\u1886\\u18AB-\\u18AF\\u18F6-\\u18FF\\u191F-\\u194F\\u196E\\u196F\\u1975-\\u197F\\u19AC-\\u19AF\\u19CA-\\u19FF\\u1A17-\\u1A1F\\u1A55-\\u1B04\\u1B34-\\u1B44\\u1B4C-\\u1B82\\u1BA1-\\u1BAD\\u1BB0-\\u1BB9\\u1BE6-\\u1BFF\\u1C24-\\u1C4C\\u1C50-\\u1C59\\u1C78-\\u1CE8\\u1CF7-\\u1CF9\\u1CFB-\\u2134\\u2139-\\u2D2F\\u2D68-\\u2D7F\\u2D97-\\u2D9F\\u2DDF-\\u3005\\u3007-\\u303B\\u303D-\\u3040\\u3097-\\u309E\\u30FB-\\u30FE\\u3100-\\u3104\\u318F-\\u319F\\u31C0-\\u31EF\\u3200-\\u33FF\\u4DC0-\\u4DFF\\u9FFD-\\u9FFF\\uA48D-\\uA4CF\\uA4F8-\\uA4FF\\uA60C-\\uA60F\\uA620-\\uA629\\uA62C-\\uA66D\\uA66F-\\uA69F\\uA6E6-\\uA78E\\uA790-\\uA7F6\\uA7F8-\\uA7FA\\uA823-\\uA83F\\uA874-\\uA881\\uA8B4-\\uA8F1\\uA8F8-\\uA8FA\\uA8FF-\\uA909\\uA926-\\uA92F\\uA947-\\uA95F\\uA97D-\\uA983\\uA9B3-\\uA9DF\\uA9E5\\uA9E6\\uA9F0-\\uA9F9\\uAA29-\\uAA3F\\uAA4C-\\uAA5F\\uAA77-\\uAA79\\uAA7B-\\uAA7D\\uAAB2-\\uAAB4\\uAAB7\\uAAB8\\uAABE\\uAABF\\uAAC3-\\uAADA\\uAADD-\\uAADF\\uAAEB-\\uAAF1\\uAAF3-\\uAB00\\uAB07\\uAB08\\uAB0F\\uAB10\\uAB17-\\uAB1F\\uAB2F-\\uABBF\\uABE3-\\uABFF\\uD7A4-\\uD7AF\\uD7C7-\\uD7CA\\uD7FC-\\uF8FF\\uFA6E\\uFA6F\\uFADA-\\uFB1C\\uFBB2-\\uFBD2\\uFD3E-\\uFD4F\\uFD90\\uFD91\\uFDC8-\\uFDEF\\uFDFC-\\uFE6F\\uFEFD-\\uFF65\\uFF9E\\uFF9F\\uFFBF-\\uFFC1\\uFFC8\\uFFC9\\uFFD0\\uFFD1\\uFFD8\\uFFD9\\uFFDD-\\uFFFF',
		'PN' => '\\u0000-/\\:-\\u00B1\\u00B4-\\u00B8\\u00BA\\u00BB\\u00BF-\\u065F\\u066A-\\u06EF\\u06FA-\\u07BF\\u07CA-\\u0965\\u0970-\\u09E5\\u09F0-\\u09F3\\u09FA-\\u0A65\\u0A70-\\u0AE5\\u0AF0-\\u0B65\\u0B70\\u0B71\\u0B78-\\u0BE5\\u0BF3-\\u0C65\\u0C70-\\u0C77\\u0C7F-\\u0CE5\\u0CF0-\\u0D57\\u0D5F-\\u0D65\\u0D79-\\u0DE5\\u0DF0-\\u0E4F\\u0E5A-\\u0ECF\\u0EDA-\\u0F1F\\u0F34-\\u103F\\u104A-\\u108F\\u109A-\\u1368\\u137D-\\u16ED\\u16F1-\\u17DF\\u17EA-\\u17EF\\u17FA-\\u180F\\u181A-\\u1945\\u1950-\\u19CF\\u19DB-\\u1A7F\\u1A8A-\\u1A8F\\u1A9A-\\u1B4F\\u1B5A-\\u1BAF\\u1BBA-\\u1C3F\\u1C4A-\\u1C4F\\u1C5A-\\u206F\\u2071-\\u2073\\u207A-\\u207F\\u208A-\\u214F\\u2183\\u2184\\u218A-\\u245F\\u249C-\\u24E9\\u2500-\\u2775\\u2794-\\u2CFC\\u2CFE-\\u3006\\u3008-\\u3020\\u302A-\\u3037\\u303B-\\u3191\\u3196-\\u321F\\u322A-\\u3247\\u3260-\\u327F\\u328A-\\u32B0\\u32C0-\\uA61F\\uA62A-\\uA6E5\\uA6F0-\\uA82F\\uA836-\\uA8CF\\uA8DA-\\uA8FF\\uA90A-\\uA9CF\\uA9DA-\\uA9EF\\uA9FA-\\uAA4F\\uAA5A-\\uABEF\\uABFA-\\uFF0F\\uFF1A-\\uFFFF',
		'PNd' => '\\u0000-/\\:-\\u065F\\u066A-\\u06EF\\u06FA-\\u07BF\\u07CA-\\u0965\\u0970-\\u09E5\\u09F0-\\u0A65\\u0A70-\\u0AE5\\u0AF0-\\u0B65\\u0B70-\\u0BE5\\u0BF0-\\u0C65\\u0C70-\\u0CE5\\u0CF0-\\u0D65\\u0D70-\\u0DE5\\u0DF0-\\u0E4F\\u0E5A-\\u0ECF\\u0EDA-\\u0F1F\\u0F2A-\\u103F\\u104A-\\u108F\\u109A-\\u17DF\\u17EA-\\u180F\\u181A-\\u1945\\u1950-\\u19CF\\u19DA-\\u1A7F\\u1A8A-\\u1A8F\\u1A9A-\\u1B4F\\u1B5A-\\u1BAF\\u1BBA-\\u1C3F\\u1C4A-\\u1C4F\\u1C5A-\\uA61F\\uA62A-\\uA8CF\\uA8DA-\\uA8FF\\uA90A-\\uA9CF\\uA9DA-\\uA9EF\\uA9FA-\\uAA4F\\uAA5A-\\uABEF\\uABFA-\\uFF0F\\uFF1A-\\uFFFF',
		'PNl' => '\\u0000-\\u16ED\\u16F1-\\u215F\\u2183\\u2184\\u2189-\\u3006\\u3008-\\u3020\\u302A-\\u3037\\u303B-\\uA6E5\\uA6F0-\\uFFFF',
		'PNo' => '\\u0000-\\u00B1\\u00B4-\\u00B8\\u00BA\\u00BB\\u00BF-\\u09F3\\u09FA-\\u0B71\\u0B78-\\u0BEF\\u0BF3-\\u0C77\\u0C7F-\\u0D57\\u0D5F-\\u0D6F\\u0D79-\\u0F29\\u0F34-\\u1368\\u137D-\\u17EF\\u17FA-\\u19D9\\u19DB-\\u206F\\u2071-\\u2073\\u207A-\\u207F\\u208A-\\u214F\\u2160-\\u2188\\u218A-\\u245F\\u249C-\\u24E9\\u2500-\\u2775\\u2794-\\u2CFC\\u2CFE-\\u3191\\u3196-\\u321F\\u322A-\\u3247\\u3260-\\u327F\\u328A-\\u32B0\\u32C0-\\uA82F\\uA836-\\uFFFF',
		'PP' => '\\u0000- 0-9\\<-\\>A-Z`-z~-\\u00A0\\u00A2-\\u00A6\\u00A8-\\u00AA\\u00AC-\\u00B5\\u00B8-\\u00BA\\u00BC-\\u00BE\\u00C0-\\u037D\\u037F-\\u0386\\u0388-\\u0559\\u0560-\\u0588\\u058B-\\u05BD\\u05C1\\u05C2\\u05C4\\u05C5\\u05C7-\\u05F2\\u05F5-\\u0608\\u060E-\\u061A\\u061C\\u061D\\u0620-\\u0669\\u066E-\\u06D3\\u06D5-\\u06FF\\u070E-\\u07F6\\u07FA-\\u082F\\u083F-\\u085D\\u085F-\\u0963\\u0966-\\u096F\\u0971-\\u09FC\\u09FE-\\u0A75\\u0A77-\\u0AEF\\u0AF1-\\u0C76\\u0C78-\\u0C83\\u0C85-\\u0DF3\\u0DF5-\\u0E4E\\u0E50-\\u0E59\\u0E5C-\\u0F03\\u0F15-\\u0F39\\u0F3E-\\u0F84\\u0F86-\\u0FCF\\u0FD5-\\u0FD8\\u0FDB-\\u1049\\u1050-\\u10FA\\u10FC-\\u135F\\u1369-\\u13FF\\u1401-\\u166D\\u166F-\\u169A\\u169D-\\u16EA\\u16EE-\\u1734\\u1737-\\u17D3\\u17DB-\\u17FF\\u180B-\\u1943\\u1946-\\u1A1D\\u1A20-\\u1A9F\\u1AAE-\\u1B59\\u1B61-\\u1BFB\\u1C00-\\u1C3A\\u1C40-\\u1C7D\\u1C80-\\u1CBF\\u1CC8-\\u1CD2\\u1CD4-\\u200F\\u2028-\\u202F\\u205F-\\u207C\\u207F-\\u208C\\u208F-\\u2307\\u230C-\\u2328\\u232B-\\u2767\\u2776-\\u27C4\\u27C7-\\u27E5\\u27F0-\\u2982\\u2999-\\u29D7\\u29DC-\\u29FB\\u29FE-\\u2CF8\\u2D00-\\u2D6F\\u2D71-\\u2DFF\\u2E50\\u2E51\\u2E53-\\u3000\\u3004-\\u3007\\u3012\\u3013\\u3020-\\u302F\\u3031-\\u303C\\u303E-\\u309F\\u30A1-\\u30FA\\u30FC-\\uA4FD\\uA500-\\uA60C\\uA610-\\uA672\\uA674-\\uA67D\\uA67F-\\uA6F1\\uA6F8-\\uA873\\uA878-\\uA8CD\\uA8D0-\\uA8F7\\uA8FD-\\uA92D\\uA930-\\uA95E\\uA960-\\uA9C0\\uA9CE-\\uA9DD\\uA9E0-\\uAA5B\\uAA60-\\uAADD\\uAAE0-\\uAAEF\\uAAF2-\\uABEA\\uABEC-\\uFD3D\\uFD40-\\uFE0F\\uFE1A-\\uFE2F\\uFE64-\\uFE67\\uFE6C-\\uFF00\\uFF10-\\uFF19\\uFF1C-\\uFF1E\\uFF21-\\uFF3A\\uFF40-\\uFF5A\\uFF66-\\uFFFF',
		'PPc' => '\\u0000-\\^`-\\u203E\\u2041-\\u2053\\u2055-\\uFE32\\uFE35-\\uFE4C\\uFE50-\\uFF3E\\uFF40-\\uFFFF',
		'PPd' => '\\u0000-,\\.-\\u0589\\u058B-\\u05BD\\u05BF-\\u13FF\\u1401-\\u1805\\u1807-\\u200F\\u2016-\\u2E16\\u2E18\\u2E19\\u2E1B-\\u2E39\\u2E3C-\\u2E3F\\u2E41-\\u301B\\u301D-\\u302F\\u3031-\\u309F\\u30A1-\\uFE30\\uFE33-\\uFE57\\uFE59-\\uFE62\\uFE64-\\uFF0C\\uFF0E-\\uFFFF',
		'PPe' => '\\u0000-\\(\\*-\\\\\\^-\\|~-\\u0F3A\\u0F3E-\\u169B\\u169D-\\u2045\\u2047-\\u207D\\u207F-\\u208D\\u208F-\\u2308\\u230C-\\u2329\\u232B-\\u2768\\u2776-\\u27C5\\u27C7-\\u27E6\\u27F0-\\u2983\\u2999-\\u29D8\\u29DC-\\u29FC\\u29FE-\\u2E22\\u2E2A-\\u3008\\u3012-\\u3014\\u301C\\u301D\\u3020-\\uFD3D\\uFD3F-\\uFE17\\uFE19-\\uFE35\\uFE45-\\uFE47\\uFE49-\\uFE59\\uFE5F-\\uFF08\\uFF0A-\\uFF3C\\uFF3E-\\uFF5C\\uFF5E\\uFF5F\\uFF61\\uFF62\\uFF64-\\uFFFF',
		'PPf' => '\\u0000-\\u00BA\\u00BC-\\u2018\\u201A-\\u201C\\u201E-\\u2039\\u203B-\\u2E02\\u2E06-\\u2E09\\u2E0B\\u2E0C\\u2E0E-\\u2E1C\\u2E1E-\\u2E20\\u2E22-\\uFFFF',
		'PPi' => '\\u0000-\\u00AA\\u00AC-\\u2017\\u2019\\u201A\\u201D\\u201E\\u2020-\\u2038\\u203A-\\u2E01\\u2E05-\\u2E08\\u2E0A\\u2E0B\\u2E0D-\\u2E1B\\u2E1D-\\u2E1F\\u2E21-\\uFFFF',
		'PPo' => '\\u0000- \\(\\)0-9\\<-\\>A-\\[\\]-\\u00A0\\u00A2-\\u00A6\\u00A8-\\u00B5\\u00B8-\\u00BE\\u00C0-\\u037D\\u037F-\\u0386\\u0388-\\u0559\\u0560-\\u0588\\u058A-\\u05BF\\u05C1\\u05C2\\u05C4\\u05C5\\u05C7-\\u05F2\\u05F5-\\u0608\\u060E-\\u061A\\u061C\\u061D\\u0620-\\u0669\\u066E-\\u06D3\\u06D5-\\u06FF\\u070E-\\u07F6\\u07FA-\\u082F\\u083F-\\u085D\\u085F-\\u0963\\u0966-\\u096F\\u0971-\\u09FC\\u09FE-\\u0A75\\u0A77-\\u0AEF\\u0AF1-\\u0C76\\u0C78-\\u0C83\\u0C85-\\u0DF3\\u0DF5-\\u0E4E\\u0E50-\\u0E59\\u0E5C-\\u0F03\\u0F15-\\u0F84\\u0F86-\\u0FCF\\u0FD5-\\u0FD8\\u0FDB-\\u1049\\u1050-\\u10FA\\u10FC-\\u135F\\u1369-\\u166D\\u166F-\\u16EA\\u16EE-\\u1734\\u1737-\\u17D3\\u17DB-\\u17FF\\u180B-\\u1943\\u1946-\\u1A1D\\u1A20-\\u1A9F\\u1AAE-\\u1B59\\u1B61-\\u1BFB\\u1C00-\\u1C3A\\u1C40-\\u1C7D\\u1C80-\\u1CBF\\u1CC8-\\u1CD2\\u1CD4-\\u2015\\u2018-\\u201F\\u2028-\\u202F\\u2039\\u203A\\u203F\\u2040\\u2044-\\u2046\\u205F-\\u2CF8\\u2D00-\\u2D6F\\u2D71-\\u2DFF\\u2E02-\\u2E05\\u2E09\\u2E0A\\u2E0C\\u2E0D\\u2E1C\\u2E1D\\u2E20-\\u2E29\\u2E3A\\u2E3B\\u2E50\\u2E51\\u2E53-\\u3000\\u3004-\\u303C\\u303E-\\u30FA\\u30FC-\\uA4FD\\uA500-\\uA60C\\uA610-\\uA672\\uA674-\\uA67D\\uA67F-\\uA6F1\\uA6F8-\\uA873\\uA878-\\uA8CD\\uA8D0-\\uA8F7\\uA8FD-\\uA92D\\uA930-\\uA95E\\uA960-\\uA9C0\\uA9CE-\\uA9DD\\uA9E0-\\uAA5B\\uAA60-\\uAADD\\uAAE0-\\uAAEF\\uAAF2-\\uABEA\\uABEC-\\uFE0F\\uFE17\\uFE18\\uFE1A-\\uFE2F\\uFE31-\\uFE44\\uFE47\\uFE48\\uFE4D-\\uFE4F\\uFE58-\\uFE5E\\uFE62-\\uFE67\\uFE6C-\\uFF00\\uFF08\\uFF09\\uFF10-\\uFF19\\uFF1C-\\uFF1E\\uFF21-\\uFF3B\\uFF3D-\\uFF60\\uFF62\\uFF63\\uFF66-\\uFFFF',
		'PPs' => '\\u0000-\'\\)-Z\\\\-z\\|-\\u0F39\\u0F3D-\\u169A\\u169C-\\u2019\\u201B-\\u201D\\u201F-\\u2044\\u2046-\\u207C\\u207E-\\u208C\\u208E-\\u2307\\u230B-\\u2328\\u232A-\\u2767\\u2775-\\u27C4\\u27C6-\\u27E5\\u27EF-\\u2982\\u2998-\\u29D7\\u29DB-\\u29FB\\u29FD-\\u2E21\\u2E29-\\u2E41\\u2E43-\\u3007\\u3011-\\u3013\\u301B\\u301C\\u301E-\\uFD3E\\uFD40-\\uFE16\\uFE18-\\uFE34\\uFE44-\\uFE46\\uFE48-\\uFE58\\uFE5E-\\uFF07\\uFF09-\\uFF3A\\uFF3C-\\uFF5A\\uFF5C-\\uFF5E\\uFF60\\uFF61\\uFF63-\\uFFFF',
		'PS' => '\\u0000-\\#%-\\*,-;\\?-\\]a-\\{\\u007F-\\u00A1\\u00AA\\u00AB\\u00B2\\u00B3\\u00B5-\\u00B7\\u00B9-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u02C1\\u02C6-\\u02D1\\u02E0-\\u02E4\\u0300-\\u0374\\u0376-\\u0383\\u0386-\\u03F5\\u03F7-\\u0481\\u0483-\\u058C\\u0590-\\u0605\\u0609\\u060A\\u060C\\u060D\\u0610-\\u06DD\\u06DF-\\u06E8\\u06EA-\\u06FC\\u06FF-\\u07F5\\u07F7-\\u07FD\\u0800-\\u09F1\\u09F4-\\u09F9\\u09FC-\\u0AF0\\u0AF2-\\u0B6F\\u0B71-\\u0BF2\\u0BFB-\\u0C7E\\u0C80-\\u0D4E\\u0D50-\\u0D78\\u0D7A-\\u0E3E\\u0E40-\\u0F00\\u0F04-\\u0F12\\u0F18\\u0F19\\u0F20-\\u0F33\\u0F39-\\u0FBD\\u0FD0-\\u0FD4\\u0FD9-\\u109D\\u10A0-\\u138F\\u139A-\\u166C\\u166E-\\u17DA\\u17DC-\\u193F\\u1941-\\u19DD\\u1A00-\\u1B60\\u1B6B-\\u1B73\\u1B7D-\\u1FBC\\u1FC2-\\u1FCC\\u1FD0-\\u1FDC\\u1FE0-\\u1FEC\\u1FF0-\\u1FFC\\u1FFF-\\u2043\\u2045-\\u2051\\u2053-\\u2079\\u207D-\\u2089\\u208D-\\u209F\\u20C0-\\u20FF\\u210A-\\u2113\\u2119-\\u211D\\u212A-\\u212D\\u212F-\\u2139\\u213C-\\u213F\\u2145-\\u2149\\u2150-\\u2189\\u218C-\\u218F\\u2308-\\u230B\\u2329\\u232A\\u2427-\\u243F\\u244B-\\u249B\\u24EA-\\u24FF\\u2768-\\u2793\\u27C5\\u27C6\\u27E6-\\u27EF\\u2983-\\u2998\\u29D8-\\u29DB\\u29FC\\u29FD\\u2B74\\u2B75\\u2C00-\\u2CE4\\u2CEB-\\u2E4F\\u2E52-\\u2E7F\\u2EF4-\\u2EFF\\u2FD6-\\u2FEF\\u2FFC-\\u3003\\u3005-\\u3011\\u3014-\\u301F\\u3021-\\u3035\\u3038-\\u303D\\u3040-\\u309A\\u309D-\\u318F\\u3192-\\u3195\\u31A0-\\u31BF\\u31E4-\\u31FF\\u321F-\\u3229\\u3248-\\u324F\\u3251-\\u325F\\u3280-\\u3289\\u32B1-\\u32BF\\u3400-\\u4DBF\\u4E00-\\uA48F\\uA4C7-\\uA6FF\\uA717-\\uA71F\\uA722-\\uA788\\uA78B-\\uA827\\uA82C-\\uA835\\uA83A-\\uAA76\\uAA7A-\\uAB5A\\uAB5C-\\uAB69\\uAB6C-\\uFB28\\uFB2A-\\uFBB1\\uFBC2-\\uFDFB\\uFDFE-\\uFE61\\uFE67\\uFE68\\uFE6A-\\uFF03\\uFF05-\\uFF0A\\uFF0C-\\uFF1B\\uFF1F-\\uFF3D\\uFF41-\\uFF5B\\uFF5F-\\uFFDF\\uFFEF-\\uFFFB\\uFFFE\\uFFFF',
		'PSc' => '\\u0000-\\#%-\\u00A1\\u00A6-\\u058E\\u0590-\\u060A\\u060C-\\u07FD\\u0800-\\u09F1\\u09F4-\\u09FA\\u09FC-\\u0AF0\\u0AF2-\\u0BF8\\u0BFA-\\u0E3E\\u0E40-\\u17DA\\u17DC-\\u209F\\u20C0-\\uA837\\uA839-\\uFDFB\\uFDFD-\\uFE68\\uFE6A-\\uFF03\\uFF05-\\uFFDF\\uFFE2-\\uFFE4\\uFFE7-\\uFFFF',
		'PSk' => '\\u0000-\\]a-\\u00A7\\u00A9-\\u00AE\\u00B0-\\u00B3\\u00B5-\\u00B7\\u00B9-\\u02C1\\u02C6-\\u02D1\\u02E0-\\u02E4\\u0300-\\u0374\\u0376-\\u0383\\u0386-\\u1FBC\\u1FC2-\\u1FCC\\u1FD0-\\u1FDC\\u1FE0-\\u1FEC\\u1FF0-\\u1FFC\\u1FFF-\\u309A\\u309D-\\uA6FF\\uA717-\\uA71F\\uA722-\\uA788\\uA78B-\\uAB5A\\uAB5C-\\uAB69\\uAB6C-\\uFBB1\\uFBC2-\\uFF3D\\uFF41-\\uFFE2\\uFFE4-\\uFFFF',
		'PSm' => '\\u0000-\\*,-;\\?-\\{\\u007F-\\u00AB\\u00AD-\\u00B0\\u00B2-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u03F5\\u03F7-\\u0605\\u0609-\\u2043\\u2045-\\u2051\\u2053-\\u2079\\u207D-\\u2089\\u208D-\\u2117\\u2119-\\u213F\\u2145-\\u214A\\u214C-\\u218F\\u2195-\\u2199\\u219C-\\u219F\\u21A1\\u21A2\\u21A4\\u21A5\\u21A7-\\u21AD\\u21AF-\\u21CD\\u21D0\\u21D1\\u21D5-\\u21F3\\u2300-\\u231F\\u2322-\\u237B\\u237D-\\u239A\\u23B4-\\u23DB\\u23E2-\\u25B6\\u25B8-\\u25C0\\u25C2-\\u25F7\\u2600-\\u266E\\u2670-\\u27BF\\u27C5\\u27C6\\u27E6-\\u27EF\\u2800-\\u28FF\\u2983-\\u2998\\u29D8-\\u29DB\\u29FC\\u29FD\\u2B00-\\u2B2F\\u2B45\\u2B46\\u2B4D-\\uFB28\\uFB2A-\\uFE61\\uFE67-\\uFF0A\\uFF0C-\\uFF1B\\uFF1F-\\uFF5B\\uFF5F-\\uFFE1\\uFFE3-\\uFFE8\\uFFED-\\uFFFF',
		'PSo' => '\\u0000-\\u00A5\\u00A7\\u00A8\\u00AA-\\u00AD\\u00B1-\\u0481\\u0483-\\u058C\\u058F-\\u060D\\u0610-\\u06DD\\u06DF-\\u06E8\\u06EA-\\u06FC\\u06FF-\\u07F5\\u07F7-\\u09F9\\u09FB-\\u0B6F\\u0B71-\\u0BF2\\u0BFB-\\u0C7E\\u0C80-\\u0D4E\\u0D50-\\u0D78\\u0D7A-\\u0F00\\u0F04-\\u0F12\\u0F18\\u0F19\\u0F20-\\u0F33\\u0F39-\\u0FBD\\u0FD0-\\u0FD4\\u0FD9-\\u109D\\u10A0-\\u138F\\u139A-\\u166C\\u166E-\\u193F\\u1941-\\u19DD\\u1A00-\\u1B60\\u1B6B-\\u1B73\\u1B7D-\\u20FF\\u210A-\\u2113\\u2118-\\u211D\\u212A-\\u212D\\u212F-\\u2139\\u213C-\\u2149\\u2150-\\u2189\\u218C-\\u2194\\u219A\\u219B\\u21CE\\u21CF\\u21F4-\\u22FF\\u2308-\\u230B\\u2320\\u2321\\u2329\\u232A\\u239B-\\u23B3\\u23DC-\\u23E1\\u2427-\\u243F\\u244B-\\u249B\\u24EA-\\u24FF\\u25F8-\\u25FF\\u2768-\\u2793\\u27C0-\\u27FF\\u2900-\\u2AFF\\u2B30-\\u2B44\\u2B47-\\u2B4C\\u2B74\\u2B75\\u2C00-\\u2CE4\\u2CEB-\\u2E4F\\u2E52-\\u2E7F\\u2EF4-\\u2EFF\\u2FD6-\\u2FEF\\u2FFC-\\u3003\\u3005-\\u3011\\u3014-\\u301F\\u3021-\\u3035\\u3038-\\u303D\\u3040-\\u318F\\u3192-\\u3195\\u31A0-\\u31BF\\u31E4-\\u31FF\\u321F-\\u3229\\u3248-\\u324F\\u3251-\\u325F\\u3280-\\u3289\\u32B1-\\u32BF\\u3400-\\u4DBF\\u4E00-\\uA48F\\uA4C7-\\uA827\\uA82C-\\uA835\\uA83A-\\uAA76\\uAA7A-\\uFDFC\\uFDFE-\\uFFE3\\uFFE5-\\uFFE7\\uFFE9-\\uFFEC\\uFFEF-\\uFFFB\\uFFFE\\uFFFF',
		'PZ' => '\\u0000-\\u001F\\!-\\u009F\\u00A1-\\u167F\\u1681-\\u1FFF\\u200B-\\u2027\\u202A-\\u202E\\u2030-\\u205E\\u2060-\\u2FFF\\u3001-\\uFFFF',
		'PZl' => '\\u0000-\\u2027\\u2029-\\uFFFF',
		'PZp' => '\\u0000-\\u2028\\u202A-\\uFFFF',
		'PZs' => '\\u0000-\\u001F\\!-\\u009F\\u00A1-\\u167F\\u1681-\\u1FFF\\u200B-\\u202E\\u2030-\\u205E\\u2060-\\u2FFF\\u3001-\\uFFFF',
		'pL' => 'A-Za-z\\u00AA\\u00B5\\u00BA\\u00C0-\\u00D6\\u00D8-\\u00F6\\u00F8-\\u02C1\\u02C6-\\u02D1\\u02E0-\\u02E4\\u02EC\\u02EE\\u0370-\\u0374\\u0376\\u0377\\u037A-\\u037D\\u037F\\u0386\\u0388-\\u038A\\u038C\\u038E-\\u03A1\\u03A3-\\u03F5\\u03F7-\\u0481\\u048A-\\u052F\\u0531-\\u0556\\u0559\\u0560-\\u0588\\u05D0-\\u05EA\\u05EF-\\u05F2\\u0620-\\u064A\\u066E\\u066F\\u0671-\\u06D3\\u06D5\\u06E5\\u06E6\\u06EE\\u06EF\\u06FA-\\u06FC\\u06FF\\u0710\\u0712-\\u072F\\u074D-\\u07A5\\u07B1\\u07CA-\\u07EA\\u07F4\\u07F5\\u07FA\\u0800-\\u0815\\u081A\\u0824\\u0828\\u0840-\\u0858\\u0860-\\u086A\\u08A0-\\u08B4\\u08B6-\\u08C7\\u0904-\\u0939\\u093D\\u0950\\u0958-\\u0961\\u0971-\\u0980\\u0985-\\u098C\\u098F\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BD\\u09CE\\u09DC\\u09DD\\u09DF-\\u09E1\\u09F0\\u09F1\\u09FC\\u0A05-\\u0A0A\\u0A0F\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32\\u0A33\\u0A35\\u0A36\\u0A38\\u0A39\\u0A59-\\u0A5C\\u0A5E\\u0A72-\\u0A74\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2\\u0AB3\\u0AB5-\\u0AB9\\u0ABD\\u0AD0\\u0AE0\\u0AE1\\u0AF9\\u0B05-\\u0B0C\\u0B0F\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32\\u0B33\\u0B35-\\u0B39\\u0B3D\\u0B5C\\u0B5D\\u0B5F-\\u0B61\\u0B71\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99\\u0B9A\\u0B9C\\u0B9E\\u0B9F\\u0BA3\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BD0\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C39\\u0C3D\\u0C58-\\u0C5A\\u0C60\\u0C61\\u0C80\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBD\\u0CDE\\u0CE0\\u0CE1\\u0CF1\\u0CF2\\u0D04-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D3A\\u0D3D\\u0D4E\\u0D54-\\u0D56\\u0D5F-\\u0D61\\u0D7A-\\u0D7F\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0E01-\\u0E30\\u0E32\\u0E33\\u0E40-\\u0E46\\u0E81\\u0E82\\u0E84\\u0E86-\\u0E8A\\u0E8C-\\u0EA3\\u0EA5\\u0EA7-\\u0EB0\\u0EB2\\u0EB3\\u0EBD\\u0EC0-\\u0EC4\\u0EC6\\u0EDC-\\u0EDF\\u0F00\\u0F40-\\u0F47\\u0F49-\\u0F6C\\u0F88-\\u0F8C\\u1000-\\u102A\\u103F\\u1050-\\u1055\\u105A-\\u105D\\u1061\\u1065\\u1066\\u106E-\\u1070\\u1075-\\u1081\\u108E\\u10A0-\\u10C5\\u10C7\\u10CD\\u10D0-\\u10FA\\u10FC-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u1380-\\u138F\\u13A0-\\u13F5\\u13F8-\\u13FD\\u1401-\\u166C\\u166F-\\u167F\\u1681-\\u169A\\u16A0-\\u16EA\\u16F1-\\u16F8\\u1700-\\u170C\\u170E-\\u1711\\u1720-\\u1731\\u1740-\\u1751\\u1760-\\u176C\\u176E-\\u1770\\u1780-\\u17B3\\u17D7\\u17DC\\u1820-\\u1878\\u1880-\\u1884\\u1887-\\u18A8\\u18AA\\u18B0-\\u18F5\\u1900-\\u191E\\u1950-\\u196D\\u1970-\\u1974\\u1980-\\u19AB\\u19B0-\\u19C9\\u1A00-\\u1A16\\u1A20-\\u1A54\\u1AA7\\u1B05-\\u1B33\\u1B45-\\u1B4B\\u1B83-\\u1BA0\\u1BAE\\u1BAF\\u1BBA-\\u1BE5\\u1C00-\\u1C23\\u1C4D-\\u1C4F\\u1C5A-\\u1C7D\\u1C80-\\u1C88\\u1C90-\\u1CBA\\u1CBD-\\u1CBF\\u1CE9-\\u1CEC\\u1CEE-\\u1CF3\\u1CF5\\u1CF6\\u1CFA\\u1D00-\\u1DBF\\u1E00-\\u1F15\\u1F18-\\u1F1D\\u1F20-\\u1F45\\u1F48-\\u1F4D\\u1F50-\\u1F57\\u1F59\\u1F5B\\u1F5D\\u1F5F-\\u1F7D\\u1F80-\\u1FB4\\u1FB6-\\u1FBC\\u1FBE\\u1FC2-\\u1FC4\\u1FC6-\\u1FCC\\u1FD0-\\u1FD3\\u1FD6-\\u1FDB\\u1FE0-\\u1FEC\\u1FF2-\\u1FF4\\u1FF6-\\u1FFC\\u2071\\u207F\\u2090-\\u209C\\u2102\\u2107\\u210A-\\u2113\\u2115\\u2119-\\u211D\\u2124\\u2126\\u2128\\u212A-\\u212D\\u212F-\\u2139\\u213C-\\u213F\\u2145-\\u2149\\u214E\\u2183\\u2184\\u2C00-\\u2C2E\\u2C30-\\u2C5E\\u2C60-\\u2CE4\\u2CEB-\\u2CEE\\u2CF2\\u2CF3\\u2D00-\\u2D25\\u2D27\\u2D2D\\u2D30-\\u2D67\\u2D6F\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u2E2F\\u3005\\u3006\\u3031-\\u3035\\u303B\\u303C\\u3041-\\u3096\\u309D-\\u309F\\u30A1-\\u30FA\\u30FC-\\u30FF\\u3105-\\u312F\\u3131-\\u318E\\u31A0-\\u31BF\\u31F0-\\u31FF\\u3400-\\u4DBF\\u4E00-\\u9FFC\\uA000-\\uA48C\\uA4D0-\\uA4FD\\uA500-\\uA60C\\uA610-\\uA61F\\uA62A\\uA62B\\uA640-\\uA66E\\uA67F-\\uA69D\\uA6A0-\\uA6E5\\uA717-\\uA71F\\uA722-\\uA788\\uA78B-\\uA7BF\\uA7C2-\\uA7CA\\uA7F5-\\uA801\\uA803-\\uA805\\uA807-\\uA80A\\uA80C-\\uA822\\uA840-\\uA873\\uA882-\\uA8B3\\uA8F2-\\uA8F7\\uA8FB\\uA8FD\\uA8FE\\uA90A-\\uA925\\uA930-\\uA946\\uA960-\\uA97C\\uA984-\\uA9B2\\uA9CF\\uA9E0-\\uA9E4\\uA9E6-\\uA9EF\\uA9FA-\\uA9FE\\uAA00-\\uAA28\\uAA40-\\uAA42\\uAA44-\\uAA4B\\uAA60-\\uAA76\\uAA7A\\uAA7E-\\uAAAF\\uAAB1\\uAAB5\\uAAB6\\uAAB9-\\uAABD\\uAAC0\\uAAC2\\uAADB-\\uAADD\\uAAE0-\\uAAEA\\uAAF2-\\uAAF4\\uAB01-\\uAB06\\uAB09-\\uAB0E\\uAB11-\\uAB16\\uAB20-\\uAB26\\uAB28-\\uAB2E\\uAB30-\\uAB5A\\uAB5C-\\uAB69\\uAB70-\\uABE2\\uAC00-\\uD7A3\\uD7B0-\\uD7C6\\uD7CB-\\uD7FB\\uF900-\\uFA6D\\uFA70-\\uFAD9\\uFB00-\\uFB06\\uFB13-\\uFB17\\uFB1D\\uFB1F-\\uFB28\\uFB2A-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40\\uFB41\\uFB43\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFB\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF21-\\uFF3A\\uFF41-\\uFF5A\\uFF66-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC',
		'pLm' => '\\u02B0-\\u02C1\\u02C6-\\u02D1\\u02E0-\\u02E4\\u02EC\\u02EE\\u0374\\u037A\\u0559\\u0640\\u06E5\\u06E6\\u07F4\\u07F5\\u07FA\\u081A\\u0824\\u0828\\u0971\\u0E46\\u0EC6\\u10FC\\u17D7\\u1843\\u1AA7\\u1C78-\\u1C7D\\u1D2C-\\u1D6A\\u1D78\\u1D9B-\\u1DBF\\u2071\\u207F\\u2090-\\u209C\\u2C7C\\u2C7D\\u2D6F\\u2E2F\\u3005\\u3031-\\u3035\\u303B\\u309D\\u309E\\u30FC-\\u30FE\\uA015\\uA4F8-\\uA4FD\\uA60C\\uA67F\\uA69C\\uA69D\\uA717-\\uA71F\\uA770\\uA788\\uA7F8\\uA7F9\\uA9CF\\uA9E6\\uAA70\\uAADD\\uAAF3\\uAAF4\\uAB5C-\\uAB5F\\uAB69\\uFF70\\uFF9E\\uFF9F',
		'pLo' => '\\u00AA\\u00BA\\u01BB\\u01C0-\\u01C3\\u0294\\u05D0-\\u05EA\\u05EF-\\u05F2\\u0620-\\u063F\\u0641-\\u064A\\u066E\\u066F\\u0671-\\u06D3\\u06D5\\u06EE\\u06EF\\u06FA-\\u06FC\\u06FF\\u0710\\u0712-\\u072F\\u074D-\\u07A5\\u07B1\\u07CA-\\u07EA\\u0800-\\u0815\\u0840-\\u0858\\u0860-\\u086A\\u08A0-\\u08B4\\u08B6-\\u08C7\\u0904-\\u0939\\u093D\\u0950\\u0958-\\u0961\\u0972-\\u0980\\u0985-\\u098C\\u098F\\u0990\\u0993-\\u09A8\\u09AA-\\u09B0\\u09B2\\u09B6-\\u09B9\\u09BD\\u09CE\\u09DC\\u09DD\\u09DF-\\u09E1\\u09F0\\u09F1\\u09FC\\u0A05-\\u0A0A\\u0A0F\\u0A10\\u0A13-\\u0A28\\u0A2A-\\u0A30\\u0A32\\u0A33\\u0A35\\u0A36\\u0A38\\u0A39\\u0A59-\\u0A5C\\u0A5E\\u0A72-\\u0A74\\u0A85-\\u0A8D\\u0A8F-\\u0A91\\u0A93-\\u0AA8\\u0AAA-\\u0AB0\\u0AB2\\u0AB3\\u0AB5-\\u0AB9\\u0ABD\\u0AD0\\u0AE0\\u0AE1\\u0AF9\\u0B05-\\u0B0C\\u0B0F\\u0B10\\u0B13-\\u0B28\\u0B2A-\\u0B30\\u0B32\\u0B33\\u0B35-\\u0B39\\u0B3D\\u0B5C\\u0B5D\\u0B5F-\\u0B61\\u0B71\\u0B83\\u0B85-\\u0B8A\\u0B8E-\\u0B90\\u0B92-\\u0B95\\u0B99\\u0B9A\\u0B9C\\u0B9E\\u0B9F\\u0BA3\\u0BA4\\u0BA8-\\u0BAA\\u0BAE-\\u0BB9\\u0BD0\\u0C05-\\u0C0C\\u0C0E-\\u0C10\\u0C12-\\u0C28\\u0C2A-\\u0C39\\u0C3D\\u0C58-\\u0C5A\\u0C60\\u0C61\\u0C80\\u0C85-\\u0C8C\\u0C8E-\\u0C90\\u0C92-\\u0CA8\\u0CAA-\\u0CB3\\u0CB5-\\u0CB9\\u0CBD\\u0CDE\\u0CE0\\u0CE1\\u0CF1\\u0CF2\\u0D04-\\u0D0C\\u0D0E-\\u0D10\\u0D12-\\u0D3A\\u0D3D\\u0D4E\\u0D54-\\u0D56\\u0D5F-\\u0D61\\u0D7A-\\u0D7F\\u0D85-\\u0D96\\u0D9A-\\u0DB1\\u0DB3-\\u0DBB\\u0DBD\\u0DC0-\\u0DC6\\u0E01-\\u0E30\\u0E32\\u0E33\\u0E40-\\u0E45\\u0E81\\u0E82\\u0E84\\u0E86-\\u0E8A\\u0E8C-\\u0EA3\\u0EA5\\u0EA7-\\u0EB0\\u0EB2\\u0EB3\\u0EBD\\u0EC0-\\u0EC4\\u0EDC-\\u0EDF\\u0F00\\u0F40-\\u0F47\\u0F49-\\u0F6C\\u0F88-\\u0F8C\\u1000-\\u102A\\u103F\\u1050-\\u1055\\u105A-\\u105D\\u1061\\u1065\\u1066\\u106E-\\u1070\\u1075-\\u1081\\u108E\\u1100-\\u1248\\u124A-\\u124D\\u1250-\\u1256\\u1258\\u125A-\\u125D\\u1260-\\u1288\\u128A-\\u128D\\u1290-\\u12B0\\u12B2-\\u12B5\\u12B8-\\u12BE\\u12C0\\u12C2-\\u12C5\\u12C8-\\u12D6\\u12D8-\\u1310\\u1312-\\u1315\\u1318-\\u135A\\u1380-\\u138F\\u1401-\\u166C\\u166F-\\u167F\\u1681-\\u169A\\u16A0-\\u16EA\\u16F1-\\u16F8\\u1700-\\u170C\\u170E-\\u1711\\u1720-\\u1731\\u1740-\\u1751\\u1760-\\u176C\\u176E-\\u1770\\u1780-\\u17B3\\u17DC\\u1820-\\u1842\\u1844-\\u1878\\u1880-\\u1884\\u1887-\\u18A8\\u18AA\\u18B0-\\u18F5\\u1900-\\u191E\\u1950-\\u196D\\u1970-\\u1974\\u1980-\\u19AB\\u19B0-\\u19C9\\u1A00-\\u1A16\\u1A20-\\u1A54\\u1B05-\\u1B33\\u1B45-\\u1B4B\\u1B83-\\u1BA0\\u1BAE\\u1BAF\\u1BBA-\\u1BE5\\u1C00-\\u1C23\\u1C4D-\\u1C4F\\u1C5A-\\u1C77\\u1CE9-\\u1CEC\\u1CEE-\\u1CF3\\u1CF5\\u1CF6\\u1CFA\\u2135-\\u2138\\u2D30-\\u2D67\\u2D80-\\u2D96\\u2DA0-\\u2DA6\\u2DA8-\\u2DAE\\u2DB0-\\u2DB6\\u2DB8-\\u2DBE\\u2DC0-\\u2DC6\\u2DC8-\\u2DCE\\u2DD0-\\u2DD6\\u2DD8-\\u2DDE\\u3006\\u303C\\u3041-\\u3096\\u309F\\u30A1-\\u30FA\\u30FF\\u3105-\\u312F\\u3131-\\u318E\\u31A0-\\u31BF\\u31F0-\\u31FF\\u3400-\\u4DBF\\u4E00-\\u9FFC\\uA000-\\uA014\\uA016-\\uA48C\\uA4D0-\\uA4F7\\uA500-\\uA60B\\uA610-\\uA61F\\uA62A\\uA62B\\uA66E\\uA6A0-\\uA6E5\\uA78F\\uA7F7\\uA7FB-\\uA801\\uA803-\\uA805\\uA807-\\uA80A\\uA80C-\\uA822\\uA840-\\uA873\\uA882-\\uA8B3\\uA8F2-\\uA8F7\\uA8FB\\uA8FD\\uA8FE\\uA90A-\\uA925\\uA930-\\uA946\\uA960-\\uA97C\\uA984-\\uA9B2\\uA9E0-\\uA9E4\\uA9E7-\\uA9EF\\uA9FA-\\uA9FE\\uAA00-\\uAA28\\uAA40-\\uAA42\\uAA44-\\uAA4B\\uAA60-\\uAA6F\\uAA71-\\uAA76\\uAA7A\\uAA7E-\\uAAAF\\uAAB1\\uAAB5\\uAAB6\\uAAB9-\\uAABD\\uAAC0\\uAAC2\\uAADB\\uAADC\\uAAE0-\\uAAEA\\uAAF2\\uAB01-\\uAB06\\uAB09-\\uAB0E\\uAB11-\\uAB16\\uAB20-\\uAB26\\uAB28-\\uAB2E\\uABC0-\\uABE2\\uAC00-\\uD7A3\\uD7B0-\\uD7C6\\uD7CB-\\uD7FB\\uF900-\\uFA6D\\uFA70-\\uFAD9\\uFB1D\\uFB1F-\\uFB28\\uFB2A-\\uFB36\\uFB38-\\uFB3C\\uFB3E\\uFB40\\uFB41\\uFB43\\uFB44\\uFB46-\\uFBB1\\uFBD3-\\uFD3D\\uFD50-\\uFD8F\\uFD92-\\uFDC7\\uFDF0-\\uFDFB\\uFE70-\\uFE74\\uFE76-\\uFEFC\\uFF66-\\uFF6F\\uFF71-\\uFF9D\\uFFA0-\\uFFBE\\uFFC2-\\uFFC7\\uFFCA-\\uFFCF\\uFFD2-\\uFFD7\\uFFDA-\\uFFDC',
		'pN' => '0-9\\u00B2\\u00B3\\u00B9\\u00BC-\\u00BE\\u0660-\\u0669\\u06F0-\\u06F9\\u07C0-\\u07C9\\u0966-\\u096F\\u09E6-\\u09EF\\u09F4-\\u09F9\\u0A66-\\u0A6F\\u0AE6-\\u0AEF\\u0B66-\\u0B6F\\u0B72-\\u0B77\\u0BE6-\\u0BF2\\u0C66-\\u0C6F\\u0C78-\\u0C7E\\u0CE6-\\u0CEF\\u0D58-\\u0D5E\\u0D66-\\u0D78\\u0DE6-\\u0DEF\\u0E50-\\u0E59\\u0ED0-\\u0ED9\\u0F20-\\u0F33\\u1040-\\u1049\\u1090-\\u1099\\u1369-\\u137C\\u16EE-\\u16F0\\u17E0-\\u17E9\\u17F0-\\u17F9\\u1810-\\u1819\\u1946-\\u194F\\u19D0-\\u19DA\\u1A80-\\u1A89\\u1A90-\\u1A99\\u1B50-\\u1B59\\u1BB0-\\u1BB9\\u1C40-\\u1C49\\u1C50-\\u1C59\\u2070\\u2074-\\u2079\\u2080-\\u2089\\u2150-\\u2182\\u2185-\\u2189\\u2460-\\u249B\\u24EA-\\u24FF\\u2776-\\u2793\\u2CFD\\u3007\\u3021-\\u3029\\u3038-\\u303A\\u3192-\\u3195\\u3220-\\u3229\\u3248-\\u324F\\u3251-\\u325F\\u3280-\\u3289\\u32B1-\\u32BF\\uA620-\\uA629\\uA6E6-\\uA6EF\\uA830-\\uA835\\uA8D0-\\uA8D9\\uA900-\\uA909\\uA9D0-\\uA9D9\\uA9F0-\\uA9F9\\uAA50-\\uAA59\\uABF0-\\uABF9\\uFF10-\\uFF19',
		'pNd' => '0-9\\u0660-\\u0669\\u06F0-\\u06F9\\u07C0-\\u07C9\\u0966-\\u096F\\u09E6-\\u09EF\\u0A66-\\u0A6F\\u0AE6-\\u0AEF\\u0B66-\\u0B6F\\u0BE6-\\u0BEF\\u0C66-\\u0C6F\\u0CE6-\\u0CEF\\u0D66-\\u0D6F\\u0DE6-\\u0DEF\\u0E50-\\u0E59\\u0ED0-\\u0ED9\\u0F20-\\u0F29\\u1040-\\u1049\\u1090-\\u1099\\u17E0-\\u17E9\\u1810-\\u1819\\u1946-\\u194F\\u19D0-\\u19D9\\u1A80-\\u1A89\\u1A90-\\u1A99\\u1B50-\\u1B59\\u1BB0-\\u1BB9\\u1C40-\\u1C49\\u1C50-\\u1C59\\uA620-\\uA629\\uA8D0-\\uA8D9\\uA900-\\uA909\\uA9D0-\\uA9D9\\uA9F0-\\uA9F9\\uAA50-\\uAA59\\uABF0-\\uABF9\\uFF10-\\uFF19',
		'pNl' => '\\u16EE-\\u16F0\\u2160-\\u2182\\u2185-\\u2188\\u3007\\u3021-\\u3029\\u3038-\\u303A\\uA6E6-\\uA6EF',
		'pNo' => '\\u00B2\\u00B3\\u00B9\\u00BC-\\u00BE\\u09F4-\\u09F9\\u0B72-\\u0B77\\u0BF0-\\u0BF2\\u0C78-\\u0C7E\\u0D58-\\u0D5E\\u0D70-\\u0D78\\u0F2A-\\u0F33\\u1369-\\u137C\\u17F0-\\u17F9\\u19DA\\u2070\\u2074-\\u2079\\u2080-\\u2089\\u2150-\\u215F\\u2189\\u2460-\\u249B\\u24EA-\\u24FF\\u2776-\\u2793\\u2CFD\\u3192-\\u3195\\u3220-\\u3229\\u3248-\\u324F\\u3251-\\u325F\\u3280-\\u3289\\u32B1-\\u32BF\\uA830-\\uA835',
		'pP' => '\\!-\\#%-\\*,-/\\:;\\?@\\[-\\]_\\{\\}\\u00A1\\u00A7\\u00AB\\u00B6\\u00B7\\u00BB\\u00BF\\u037E\\u0387\\u055A-\\u055F\\u0589\\u058A\\u05BE\\u05C0\\u05C3\\u05C6\\u05F3\\u05F4\\u0609\\u060A\\u060C\\u060D\\u061B\\u061E\\u061F\\u066A-\\u066D\\u06D4\\u0700-\\u070D\\u07F7-\\u07F9\\u0830-\\u083E\\u085E\\u0964\\u0965\\u0970\\u09FD\\u0A76\\u0AF0\\u0C77\\u0C84\\u0DF4\\u0E4F\\u0E5A\\u0E5B\\u0F04-\\u0F12\\u0F14\\u0F3A-\\u0F3D\\u0F85\\u0FD0-\\u0FD4\\u0FD9\\u0FDA\\u104A-\\u104F\\u10FB\\u1360-\\u1368\\u1400\\u166E\\u169B\\u169C\\u16EB-\\u16ED\\u1735\\u1736\\u17D4-\\u17D6\\u17D8-\\u17DA\\u1800-\\u180A\\u1944\\u1945\\u1A1E\\u1A1F\\u1AA0-\\u1AA6\\u1AA8-\\u1AAD\\u1B5A-\\u1B60\\u1BFC-\\u1BFF\\u1C3B-\\u1C3F\\u1C7E\\u1C7F\\u1CC0-\\u1CC7\\u1CD3\\u2010-\\u2027\\u2030-\\u2043\\u2045-\\u2051\\u2053-\\u205E\\u207D\\u207E\\u208D\\u208E\\u2308-\\u230B\\u2329\\u232A\\u2768-\\u2775\\u27C5\\u27C6\\u27E6-\\u27EF\\u2983-\\u2998\\u29D8-\\u29DB\\u29FC\\u29FD\\u2CF9-\\u2CFC\\u2CFE\\u2CFF\\u2D70\\u2E00-\\u2E2E\\u2E30-\\u2E4F\\u2E52\\u3001-\\u3003\\u3008-\\u3011\\u3014-\\u301F\\u3030\\u303D\\u30A0\\u30FB\\uA4FE\\uA4FF\\uA60D-\\uA60F\\uA673\\uA67E\\uA6F2-\\uA6F7\\uA874-\\uA877\\uA8CE\\uA8CF\\uA8F8-\\uA8FA\\uA8FC\\uA92E\\uA92F\\uA95F\\uA9C1-\\uA9CD\\uA9DE\\uA9DF\\uAA5C-\\uAA5F\\uAADE\\uAADF\\uAAF0\\uAAF1\\uABEB\\uFD3E\\uFD3F\\uFE10-\\uFE19\\uFE30-\\uFE52\\uFE54-\\uFE61\\uFE63\\uFE68\\uFE6A\\uFE6B\\uFF01-\\uFF03\\uFF05-\\uFF0A\\uFF0C-\\uFF0F\\uFF1A\\uFF1B\\uFF1F\\uFF20\\uFF3B-\\uFF3D\\uFF3F\\uFF5B\\uFF5D\\uFF5F-\\uFF65',
		'pPc' => '_\\u203F\\u2040\\u2054\\uFE33\\uFE34\\uFE4D-\\uFE4F\\uFF3F',
		'pPd' => '\\-\\u058A\\u05BE\\u1400\\u1806\\u2010-\\u2015\\u2E17\\u2E1A\\u2E3A\\u2E3B\\u2E40\\u301C\\u3030\\u30A0\\uFE31\\uFE32\\uFE58\\uFE63\\uFF0D',
		'pPe' => '\\)\\]\\}\\u0F3B\\u0F3D\\u169C\\u2046\\u207E\\u208E\\u2309\\u230B\\u232A\\u2769\\u276B\\u276D\\u276F\\u2771\\u2773\\u2775\\u27C6\\u27E7\\u27E9\\u27EB\\u27ED\\u27EF\\u2984\\u2986\\u2988\\u298A\\u298C\\u298E\\u2990\\u2992\\u2994\\u2996\\u2998\\u29D9\\u29DB\\u29FD\\u2E23\\u2E25\\u2E27\\u2E29\\u3009\\u300B\\u300D\\u300F\\u3011\\u3015\\u3017\\u3019\\u301B\\u301E\\u301F\\uFD3E\\uFE18\\uFE36\\uFE38\\uFE3A\\uFE3C\\uFE3E\\uFE40\\uFE42\\uFE44\\uFE48\\uFE5A\\uFE5C\\uFE5E\\uFF09\\uFF3D\\uFF5D\\uFF60\\uFF63',
		'pPf' => '\\u00BB\\u2019\\u201D\\u203A\\u2E03\\u2E05\\u2E0A\\u2E0D\\u2E1D\\u2E21',
		'pPi' => '\\u00AB\\u2018\\u201B\\u201C\\u201F\\u2039\\u2E02\\u2E04\\u2E09\\u2E0C\\u2E1C\\u2E20',
		'pPo' => '\\!-\\#%-\'\\*,\\./\\:;\\?@\\\\\\u00A1\\u00A7\\u00B6\\u00B7\\u00BF\\u037E\\u0387\\u055A-\\u055F\\u0589\\u05C0\\u05C3\\u05C6\\u05F3\\u05F4\\u0609\\u060A\\u060C\\u060D\\u061B\\u061E\\u061F\\u066A-\\u066D\\u06D4\\u0700-\\u070D\\u07F7-\\u07F9\\u0830-\\u083E\\u085E\\u0964\\u0965\\u0970\\u09FD\\u0A76\\u0AF0\\u0C77\\u0C84\\u0DF4\\u0E4F\\u0E5A\\u0E5B\\u0F04-\\u0F12\\u0F14\\u0F85\\u0FD0-\\u0FD4\\u0FD9\\u0FDA\\u104A-\\u104F\\u10FB\\u1360-\\u1368\\u166E\\u16EB-\\u16ED\\u1735\\u1736\\u17D4-\\u17D6\\u17D8-\\u17DA\\u1800-\\u1805\\u1807-\\u180A\\u1944\\u1945\\u1A1E\\u1A1F\\u1AA0-\\u1AA6\\u1AA8-\\u1AAD\\u1B5A-\\u1B60\\u1BFC-\\u1BFF\\u1C3B-\\u1C3F\\u1C7E\\u1C7F\\u1CC0-\\u1CC7\\u1CD3\\u2016\\u2017\\u2020-\\u2027\\u2030-\\u2038\\u203B-\\u203E\\u2041-\\u2043\\u2047-\\u2051\\u2053\\u2055-\\u205E\\u2CF9-\\u2CFC\\u2CFE\\u2CFF\\u2D70\\u2E00\\u2E01\\u2E06-\\u2E08\\u2E0B\\u2E0E-\\u2E16\\u2E18\\u2E19\\u2E1B\\u2E1E\\u2E1F\\u2E2A-\\u2E2E\\u2E30-\\u2E39\\u2E3C-\\u2E3F\\u2E41\\u2E43-\\u2E4F\\u2E52\\u3001-\\u3003\\u303D\\u30FB\\uA4FE\\uA4FF\\uA60D-\\uA60F\\uA673\\uA67E\\uA6F2-\\uA6F7\\uA874-\\uA877\\uA8CE\\uA8CF\\uA8F8-\\uA8FA\\uA8FC\\uA92E\\uA92F\\uA95F\\uA9C1-\\uA9CD\\uA9DE\\uA9DF\\uAA5C-\\uAA5F\\uAADE\\uAADF\\uAAF0\\uAAF1\\uABEB\\uFE10-\\uFE16\\uFE19\\uFE30\\uFE45\\uFE46\\uFE49-\\uFE4C\\uFE50-\\uFE52\\uFE54-\\uFE57\\uFE5F-\\uFE61\\uFE68\\uFE6A\\uFE6B\\uFF01-\\uFF03\\uFF05-\\uFF07\\uFF0A\\uFF0C\\uFF0E\\uFF0F\\uFF1A\\uFF1B\\uFF1F\\uFF20\\uFF3C\\uFF61\\uFF64\\uFF65',
		'pPs' => '\\(\\[\\{\\u0F3A\\u0F3C\\u169B\\u201A\\u201E\\u2045\\u207D\\u208D\\u2308\\u230A\\u2329\\u2768\\u276A\\u276C\\u276E\\u2770\\u2772\\u2774\\u27C5\\u27E6\\u27E8\\u27EA\\u27EC\\u27EE\\u2983\\u2985\\u2987\\u2989\\u298B\\u298D\\u298F\\u2991\\u2993\\u2995\\u2997\\u29D8\\u29DA\\u29FC\\u2E22\\u2E24\\u2E26\\u2E28\\u2E42\\u3008\\u300A\\u300C\\u300E\\u3010\\u3014\\u3016\\u3018\\u301A\\u301D\\uFD3F\\uFE17\\uFE35\\uFE37\\uFE39\\uFE3B\\uFE3D\\uFE3F\\uFE41\\uFE43\\uFE47\\uFE59\\uFE5B\\uFE5D\\uFF08\\uFF3B\\uFF5B\\uFF5F\\uFF62',
		'pS' => '\\$\\+\\<-\\>\\^`\\|~\\u00A2-\\u00A6\\u00A8\\u00A9\\u00AC\\u00AE-\\u00B1\\u00B4\\u00B8\\u00D7\\u00F7\\u02C2-\\u02C5\\u02D2-\\u02DF\\u02E5-\\u02EB\\u02ED\\u02EF-\\u02FF\\u0375\\u0384\\u0385\\u03F6\\u0482\\u058D-\\u058F\\u0606-\\u0608\\u060B\\u060E\\u060F\\u06DE\\u06E9\\u06FD\\u06FE\\u07F6\\u07FE\\u07FF\\u09F2\\u09F3\\u09FA\\u09FB\\u0AF1\\u0B70\\u0BF3-\\u0BFA\\u0C7F\\u0D4F\\u0D79\\u0E3F\\u0F01-\\u0F03\\u0F13\\u0F15-\\u0F17\\u0F1A-\\u0F1F\\u0F34\\u0F36\\u0F38\\u0FBE-\\u0FC5\\u0FC7-\\u0FCC\\u0FCE\\u0FCF\\u0FD5-\\u0FD8\\u109E\\u109F\\u1390-\\u1399\\u166D\\u17DB\\u1940\\u19DE-\\u19FF\\u1B61-\\u1B6A\\u1B74-\\u1B7C\\u1FBD\\u1FBF-\\u1FC1\\u1FCD-\\u1FCF\\u1FDD-\\u1FDF\\u1FED-\\u1FEF\\u1FFD\\u1FFE\\u2044\\u2052\\u207A-\\u207C\\u208A-\\u208C\\u20A0-\\u20BF\\u2100\\u2101\\u2103-\\u2106\\u2108\\u2109\\u2114\\u2116-\\u2118\\u211E-\\u2123\\u2125\\u2127\\u2129\\u212E\\u213A\\u213B\\u2140-\\u2144\\u214A-\\u214D\\u214F\\u218A\\u218B\\u2190-\\u2307\\u230C-\\u2328\\u232B-\\u2426\\u2440-\\u244A\\u249C-\\u24E9\\u2500-\\u2767\\u2794-\\u27C4\\u27C7-\\u27E5\\u27F0-\\u2982\\u2999-\\u29D7\\u29DC-\\u29FB\\u29FE-\\u2B73\\u2B76-\\u2B95\\u2B97-\\u2BFF\\u2CE5-\\u2CEA\\u2E50\\u2E51\\u2E80-\\u2E99\\u2E9B-\\u2EF3\\u2F00-\\u2FD5\\u2FF0-\\u2FFB\\u3004\\u3012\\u3013\\u3020\\u3036\\u3037\\u303E\\u303F\\u309B\\u309C\\u3190\\u3191\\u3196-\\u319F\\u31C0-\\u31E3\\u3200-\\u321E\\u322A-\\u3247\\u3250\\u3260-\\u327F\\u328A-\\u32B0\\u32C0-\\u33FF\\u4DC0-\\u4DFF\\uA490-\\uA4C6\\uA700-\\uA716\\uA720\\uA721\\uA789\\uA78A\\uA828-\\uA82B\\uA836-\\uA839\\uAA77-\\uAA79\\uAB5B\\uAB6A\\uAB6B\\uFB29\\uFBB2-\\uFBC1\\uFDFC\\uFDFD\\uFE62\\uFE64-\\uFE66\\uFE69\\uFF04\\uFF0B\\uFF1C-\\uFF1E\\uFF3E\\uFF40\\uFF5C\\uFF5E\\uFFE0-\\uFFE6\\uFFE8-\\uFFEE\\uFFFC\\uFFFD',
		'pSc' => '\\$\\u00A2-\\u00A5\\u058F\\u060B\\u07FE\\u07FF\\u09F2\\u09F3\\u09FB\\u0AF1\\u0BF9\\u0E3F\\u17DB\\u20A0-\\u20BF\\uA838\\uFDFC\\uFE69\\uFF04\\uFFE0\\uFFE1\\uFFE5\\uFFE6',
		'pSk' => '\\^`\\u00A8\\u00AF\\u00B4\\u00B8\\u02C2-\\u02C5\\u02D2-\\u02DF\\u02E5-\\u02EB\\u02ED\\u02EF-\\u02FF\\u0375\\u0384\\u0385\\u1FBD\\u1FBF-\\u1FC1\\u1FCD-\\u1FCF\\u1FDD-\\u1FDF\\u1FED-\\u1FEF\\u1FFD\\u1FFE\\u309B\\u309C\\uA700-\\uA716\\uA720\\uA721\\uA789\\uA78A\\uAB5B\\uAB6A\\uAB6B\\uFBB2-\\uFBC1\\uFF3E\\uFF40\\uFFE3',
		'pSm' => '\\+\\<-\\>\\|~\\u00AC\\u00B1\\u00D7\\u00F7\\u03F6\\u0606-\\u0608\\u2044\\u2052\\u207A-\\u207C\\u208A-\\u208C\\u2118\\u2140-\\u2144\\u214B\\u2190-\\u2194\\u219A\\u219B\\u21A0\\u21A3\\u21A6\\u21AE\\u21CE\\u21CF\\u21D2\\u21D4\\u21F4-\\u22FF\\u2320\\u2321\\u237C\\u239B-\\u23B3\\u23DC-\\u23E1\\u25B7\\u25C1\\u25F8-\\u25FF\\u266F\\u27C0-\\u27C4\\u27C7-\\u27E5\\u27F0-\\u27FF\\u2900-\\u2982\\u2999-\\u29D7\\u29DC-\\u29FB\\u29FE-\\u2AFF\\u2B30-\\u2B44\\u2B47-\\u2B4C\\uFB29\\uFE62\\uFE64-\\uFE66\\uFF0B\\uFF1C-\\uFF1E\\uFF5C\\uFF5E\\uFFE2\\uFFE9-\\uFFEC',
		'pSo' => '\\u00A6\\u00A9\\u00AE\\u00B0\\u0482\\u058D\\u058E\\u060E\\u060F\\u06DE\\u06E9\\u06FD\\u06FE\\u07F6\\u09FA\\u0B70\\u0BF3-\\u0BF8\\u0BFA\\u0C7F\\u0D4F\\u0D79\\u0F01-\\u0F03\\u0F13\\u0F15-\\u0F17\\u0F1A-\\u0F1F\\u0F34\\u0F36\\u0F38\\u0FBE-\\u0FC5\\u0FC7-\\u0FCC\\u0FCE\\u0FCF\\u0FD5-\\u0FD8\\u109E\\u109F\\u1390-\\u1399\\u166D\\u1940\\u19DE-\\u19FF\\u1B61-\\u1B6A\\u1B74-\\u1B7C\\u2100\\u2101\\u2103-\\u2106\\u2108\\u2109\\u2114\\u2116\\u2117\\u211E-\\u2123\\u2125\\u2127\\u2129\\u212E\\u213A\\u213B\\u214A\\u214C\\u214D\\u214F\\u218A\\u218B\\u2195-\\u2199\\u219C-\\u219F\\u21A1\\u21A2\\u21A4\\u21A5\\u21A7-\\u21AD\\u21AF-\\u21CD\\u21D0\\u21D1\\u21D3\\u21D5-\\u21F3\\u2300-\\u2307\\u230C-\\u231F\\u2322-\\u2328\\u232B-\\u237B\\u237D-\\u239A\\u23B4-\\u23DB\\u23E2-\\u2426\\u2440-\\u244A\\u249C-\\u24E9\\u2500-\\u25B6\\u25B8-\\u25C0\\u25C2-\\u25F7\\u2600-\\u266E\\u2670-\\u2767\\u2794-\\u27BF\\u2800-\\u28FF\\u2B00-\\u2B2F\\u2B45\\u2B46\\u2B4D-\\u2B73\\u2B76-\\u2B95\\u2B97-\\u2BFF\\u2CE5-\\u2CEA\\u2E50\\u2E51\\u2E80-\\u2E99\\u2E9B-\\u2EF3\\u2F00-\\u2FD5\\u2FF0-\\u2FFB\\u3004\\u3012\\u3013\\u3020\\u3036\\u3037\\u303E\\u303F\\u3190\\u3191\\u3196-\\u319F\\u31C0-\\u31E3\\u3200-\\u321E\\u322A-\\u3247\\u3250\\u3260-\\u327F\\u328A-\\u32B0\\u32C0-\\u33FF\\u4DC0-\\u4DFF\\uA490-\\uA4C6\\uA828-\\uA82B\\uA836\\uA837\\uA839\\uAA77-\\uAA79\\uFDFD\\uFFE4\\uFFE8\\uFFED\\uFFEE\\uFFFC\\uFFFD',
		'pZ' => ' \\u00A0\\u1680\\u2000-\\u200A\\u2028\\u2029\\u202F\\u205F\\u3000',
		'pZl' => '\\u2028',
		'pZp' => '\\u2029',
		'pZs' => ' \\u00A0\\u1680\\u2000-\\u200A\\u202F\\u205F\\u3000'
	];
}